/* FILE: mparse.c                               (D. Tottingham  03/24/91)

This is a collection of C functions that parse the control file for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

p_initialize ()                 initialize parser
p_parse_commands ()             parse commands in input stream

EXTERNAL FUNCTIONS CALLED:

dm_set_PreEventTime ()          set PreEventTime constant
dt_set_channel ()               put channel in channel list
dt_set_ChannelBlocksize ()      set the channel block size
dt_set_ChannelGain ()           set the channel gain
dt_set_ClockSource ()           set clock source
dt_set_DigitizationRate ()      set the digitization rate
dt_set_TriggerSource ()         set trigger source
dsp_set_band ()                 define a band limit
dsp_set_bufs_to_avg ()          set buffers to average
dsp_set_spectral_status ()      set spectral_enabled flag
dsp_set_dsp_status ()           set dsp_enabled flag
dsp_set_MaxCalibrationTime ()   set maximum calibration time
e_set_bell_status ()            set bell_enabled flag
f_set_NetworkNodeId ()          set network node id
f_set_pathname ()               set pathname
fr_set_block_size ()            set block size in seconds
h_set_trigger_status ()         set trigger status
hs_set_CriticalPhi ()           set critical phi
hs_set_HalfspaceVelocity ()     set halfspace velocity
l_set_location_status ()        set location_enabled flag
lx_initialize ()                initialize the lexer
lx_match ()                     match a token with the lookahead
lx_set_stream ()                set the input stream
o_set_pathname ()               set log pathname
r_set_reboot_status ()          set reboot_enabled flag
r_set_reboot_time ()            set reboot time
s_add_channel ()                add a channel to the home display
st_add_station ()               add station to station queue
st_set_authority ()             set authority code
t_set_CriticalAlpha ()          set CriticalAlpha constant
t_set_CriticalBeta ()           set CriticalBeta constant
t_set_CriticalGamma ()          set CriticalGamma constant
t_set_CriticalMu ()             set CriticalMu constant
t_set_CriticalNu ()             set CriticalNu constant
t_set_EventContinueCount ()     set EventContinuationCount constant
t_set_LTAWindow ()              set length of long-term average
t_set_MaxEventTime ()           set MaxEventTime constant
t_set_MinEventTime ()           set MinEventTime constant
t_set_STAWindow ()              set length of short-term average
t_set_Trigger ()                set trigger status for given channel
t_set_TriggerConfirmCount ()    set TriggerConfirmationCount constant
t_set_TriggerTimeLimit ()       set TriggerTimeLimit constant
t_set_trigger_status ()         set trigger_enabled flag

HISTORY:
   none

*/



/*************************************************************************
                             INCLUDE FILES

*************************************************************************/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <atldefs.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdsp.h"
#include "mdt28xx.h"
#include "mevent.h"
#include "mfile.h"
#include "mfreerun.h"
#include "mhalfsp.h"
#include "mlexer.h"
#include "mlog.h"
#include "mlocate.h"
#include "mparse.h"
#include "mreboot.h"
#include "mscrnhdr.h"
#include "mstation.h"
#include "mtrigger.h"
#include "xdetect.h"


/*=======================================================================*
 *                           process_bandlimits                          *
 *=======================================================================*/
/* Process band limits.                                                  */

PRIVATE
void process_bandlimits ()
{
   float high, low;

   lx_match (',');
   lx_match (C_BAND_LIMITS);
   lx_match ('=');
   lx_match ('{');

   while (lex_info.lookahead != '}') {
      lx_match ('(');
      if (lex_info.lookahead == C_CONSTANT)
         low = atof(lex_info.lexeme);
      lx_match (C_CONSTANT);
      lx_match (',');
      if (lex_info.lookahead == C_CONSTANT) {
         high = atof(lex_info.lexeme);
         dsp_set_band (low, high);
      }
      lx_match (C_CONSTANT);
      lx_match (')');
      if (lex_info.lookahead != '}')
         lx_match (',');
   }
   lx_match ('}');
}

/*=======================================================================*
 *                           process_station                             *
 *=======================================================================*/
/* Process station command.                                              */

PRIVATE
void process_station ()
{
   char network[MAXLEX], st_name[MAXLEX], component;
   FLAG valid_definition, trigger, display;
   unsigned int channel, gain;

   /* Initialize defaults */
   valid_definition = TRUE;
   gain = 0;
   trigger = ON;
   display = OFF;

   /* Process channel number */
   channel = atoi(lex_info.lexeme);
   lx_match (C_CONSTANT);
   lx_match (',');

   /* Process station id */
   lx_match (C_STNAME);
   lx_match ('=');
   if (lex_info.lookahead == C_STRING ||
       lex_info.lookahead == C_ID)
      strcpy (st_name, lex_info.lexeme);
   else valid_definition = FALSE;
   if (lex_info.lookahead == C_STRING)
      lx_match (C_STRING);
   else lx_match (C_ID);
   lx_match (',');

   /* Process component */
   lx_match (C_COMPONENT);
   lx_match ('=');
   if (lex_info.lookahead == C_CHARACTER ||
       lex_info.lookahead == C_ID)
      component = lex_info.lexeme[0];
   else valid_definition = FALSE;
   if (lex_info.lookahead == C_ID)
      lx_match (C_ID);
   else lx_match (C_CHARACTER);

   /* Process optional parameters */
   while (lex_info.lookahead != ';') {
      lx_match (',');
      switch (lex_info.lookahead) {
         case C_DISPLAY:
            lx_match (C_DISPLAY);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               display = (lex_info.lookahead == C_ON) ? ON : OFF;
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;

         case C_GAIN:
            lx_match (C_GAIN);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               gain = atoi(lex_info.lexeme);
            lx_match (C_CONSTANT);
            break;

         case C_TRIGGER:
            lx_match (C_TRIGGER);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               trigger = (lex_info.lookahead == C_ON) ? ON : OFF;
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;
      }
   }
   if (valid_definition) {
      if (dt_set_channel (channel, gain)) {
         st_add_station (st_name, component, channel);
         t_set_Trigger (channel, trigger);
         if (display) s_add_channel (channel);
      }
   }
}

/*=======================================================================*
 *                             p_initialize                              *
 *=======================================================================*/
/* Initialize parser.                                                    */

PUBLIC
void p_initialize ()
{
   lx_initialize();
}

/*=======================================================================*
 *                           p_parse_commands                            *
 *=======================================================================*/
/* Parse commands in input stream.                                       */

PUBLIC
void p_parse_commands (stream)
FILE * stream;
{
   FLAG spectral_status, fft_file_status;
   int hour, minute, second;
   double f;

   lx_set_stream (stream);
   lx_match (lex_info.lookahead);

   spectral_status = fft_file_status = FALSE;
   while (lex_info.lookahead != C_DONE) {
      switch (lex_info.lookahead) {
         case C_AUTHORITY:
            lx_match (C_AUTHORITY);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               st_set_authority (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_AUTO_LOCATION:
            lx_match (C_AUTO_LOCATION);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               l_set_location_status ((lex_info.lookahead == C_ON) ? ON : OFF);
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;

         case C_AUTO_REBOOT:
            lx_match (C_AUTO_REBOOT);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               r_set_reboot_status ((lex_info.lookahead == C_ON) ? ON : OFF);
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);

            if (lex_info.lookahead != ';') {
               lx_match (',');
               lx_match (C_TIME);
               lx_match ('=');
               hour = atoi(lex_info.lexeme);
               lx_match (C_CONSTANT);
               lx_match (':');
               minute = atoi(lex_info.lexeme);
               lx_match (C_CONSTANT);
               lx_match (':');
               second = atoi(lex_info.lexeme);
               lx_match (C_CONSTANT);
               r_set_reboot_time (hour, minute, second);
            }
            break;

         case C_AUTO_TRIGGER:
            lx_match (C_AUTO_TRIGGER);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               h_set_trigger_status ((lex_info.lookahead == C_ON) ? ON : OFF);
               t_set_trigger_status ((lex_info.lookahead == C_ON) ? ON : OFF);
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;

         case C_BAND_RECORDING:
            lx_match (C_BAND_RECORDING);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               fft_file_status = (lex_info.lookahead == C_ON) ? ON : OFF;
               dsp_set_fft_file_status (fft_file_status);
               lx_match (lex_info.lookahead);
               if (lex_info.lookahead == ',')
                  process_bandlimits ();
            } else lx_match (C_ON);
            break;

         case C_BUFFERS_TO_AVERAGE:
            lx_match (C_BUFFERS_TO_AVERAGE);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dsp_set_bufs_to_avg (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CALIBRATION_RECORDING:
            lx_match (C_CALIBRATION_RECORDING);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               spectral_status = (lex_info.lookahead == C_ON) ? ON : OFF;
               dsp_set_spectral_status (spectral_status);
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;

         case C_CH:
            lx_match (C_CH);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT) {
               process_station ();
            } else lx_match (C_CONSTANT);
            break;

         case C_CHANNEL_BLOCKSIZE:
            lx_match (C_CHANNEL_BLOCKSIZE);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dt_set_ChannelBlocksize (atol(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CHANNEL_GAIN:
            lx_match (C_CHANNEL_GAIN);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dt_set_ChannelGain (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CLOCK_SOURCE:
            lx_match (C_CLOCK_SOURCE);
            lx_match ('=');
            if (lex_info.lookahead == C_INTERNAL || lex_info.lookahead == C_EXTERNAL) {
               dt_set_ClockSource ((lex_info.lookahead == C_INTERNAL) ? INTERNAL_CLOCK :
                                   EXTERNAL_CLOCK);
               lx_match (lex_info.lookahead);
            } else lx_match (C_INTERNAL);
            break;

         case C_CRITICAL_ALPHA:
            lx_match (C_CRITICAL_ALPHA);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_CriticalAlpha (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CRITICAL_BETA:
            lx_match (C_CRITICAL_BETA);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_CriticalBeta (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CRITICAL_GAMMA:
            lx_match (C_CRITICAL_GAMMA);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_CriticalGamma (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CRITICAL_MU:
            lx_match (C_CRITICAL_MU);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_CriticalMu (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CRITICAL_NU:
            lx_match (C_CRITICAL_NU);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_CriticalNu (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_CRITICAL_PHI:
            lx_match (C_CRITICAL_PHI);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               hs_set_CriticalPhi (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_DIGITIZATION_RATE:
            lx_match (C_DIGITIZATION_RATE);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dt_set_DigitizationRate (f = atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_EVENT_ALERT_BELL:
            lx_match (C_EVENT_ALERT_BELL);
            lx_match ('=');
            if (lex_info.lookahead == C_ON || lex_info.lookahead == C_OFF) {
               e_set_bell_status ((lex_info.lookahead == C_ON) ? ON : OFF);
               lx_match (lex_info.lookahead);
            } else lx_match (C_ON);
            break;

         case C_EVENT_CONTINUE_COUNT:
            lx_match (C_EVENT_CONTINUE_COUNT);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_EventContinueCount (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_FREERUN_BLOCK_TIME:
            lx_match (C_FREERUN_BLOCK_TIME);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               fr_set_block_size (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_HALFSPACE_VELOCITY:
            lx_match (C_HALFSPACE_VELOCITY);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               hs_set_HalfspaceVelocity (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_LOG_PATHNAME:
            lx_match (C_LOG_PATHNAME);
            lx_match ('=');
            if (lex_info.lookahead == C_STRING) {
               printf ("WARNING: Alternate log file pathname is no longer being supported.\n");
               /* o_set_pathname (lex_info.lexeme); */
            }
            lx_match (C_STRING);
            break;

         case C_LTA_WINDOW:
            lx_match (C_LTA_WINDOW);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_LTAWindow (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_MAX_CALIBRATION_TIME:
            lx_match (C_MAX_CALIBRATION_TIME);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dsp_set_MaxCalibrationTime (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_MAX_EVENT_TIME:
            lx_match (C_MAX_EVENT_TIME);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_MaxEventTime (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_MIN_EVENT_TIME:
            lx_match (C_MIN_EVENT_TIME);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_MinEventTime (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_NETWORK:
            lx_match (C_NETWORK);
            lx_match ('=');
            if (lex_info.lookahead == C_STRING ||
                lex_info.lookahead == C_ID)
               st_set_netwname (lex_info.lexeme);
            if (lex_info.lookahead == C_STRING)
               lx_match (C_STRING);
            else lx_match (C_ID);
            break;

         case C_NETWORK_NODE_ID:
            lx_match (C_NETWORK_NODE_ID);
            lx_match ('=');
            if (lex_info.lookahead == C_STRING ||
                lex_info.lookahead == C_ID)
               f_set_NetworkNodeId (((char far *)lex_info.lexeme));
            if (lex_info.lookahead == C_STRING)
               lx_match (C_STRING);
            else lx_match (C_ID);
            break;

         case C_PATHNAME:
            lx_match (C_PATHNAME);
            lx_match ('=');
            if (lex_info.lookahead == C_STRING) {
               f_set_pathname (lex_info.lexeme);
               f_get_pathname();
            }
            lx_match (C_STRING);
            break;

         case C_PRE_EVENT_TIME:
            lx_match (C_PRE_EVENT_TIME);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               dm_set_PreEventTime (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_STA_WINDOW:
            lx_match (C_STA_WINDOW);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_STAWindow (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_TRIGGER_CONFIRM_COUNT:
            lx_match (C_TRIGGER_CONFIRM_COUNT);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_TriggerConfirmCount (atoi(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;

         case C_TRIGGER_SOURCE:
            lx_match (C_TRIGGER_SOURCE);
            lx_match ('=');
            if (lex_info.lookahead == C_INTERNAL || lex_info.lookahead == C_EXTERNAL) {
               dt_set_TriggerSource ((lex_info.lookahead == C_INTERNAL) ? INTERNAL_TRIGGER :
                                     EXTERNAL_TRIGGER);
               lx_match (lex_info.lookahead);
            } else lx_match (C_INTERNAL);
            break;

         case C_TRIGGER_TIME_LIMIT:
            lx_match (C_TRIGGER_TIME_LIMIT);
            lx_match ('=');
            if (lex_info.lookahead == C_CONSTANT)
               t_set_TriggerTimeLimit (atof(lex_info.lexeme));
            lx_match (C_CONSTANT);
            break;
      }
   lx_match (';');
   }
   dsp_set_dsp_status (spectral_status || fft_file_status);
}
